/* Skript zur Demonstration von hierarchischen Abfragen
 * Als Benutzer SCOTT ausführen
 */
 
-- Versuch einer Organigramm-Abfrage mit 4 Ebenen und Inner Joins
select e1.ename, e2.ename, e3.ename, e4.ename
  from emp e1
  join emp e2 on e1.empno = e2.mgr
  join emp e3 on e2.empno = e3.mgr
  join emp e4 on e3.empno = e4.mgr
 where e1.ename = 'KING';
  
-- Versuch einer Organigramm-Abfrage mit 4 Ebenen und Outer Joins
select e1.ename, e2.ename, e3.ename, e4.ename
  from emp e1
  left join emp e2 on e1.empno = e2.mgr
  left join emp e3 on e2.empno = e3.mgr
  left join emp e4 on e3.empno = e4.mgr
 where e1.ename = 'KING';

select e1.ename, e2.ename, e3.ename, e4.ename
  from emp e1, emp e2, emp e3, emp e4
 where e1.empno = e2.mgr (+)
   and e2.empno = e3.mgr (+)
   and e3.empno = e4.mgr (+)
   and e1.ename = 'KING';
   
-- Hierarchische Abfrage mit connect by, einfache Form
 select level, lpad('.', (level - 1) * 2, '.') || ename name
   from emp
  start with mgr is null
connect by prior empno = mgr;

-- Hierarchische Abfrage mit connect by, Einstieg eine Ebene tiefer
 select level, lpad('.', (level - 1) * 2, '.') || ename name
   from emp
  start with job = 'MANAGER'
connect by prior empno = mgr;

-- Zugriff auf den hierarchisch höher stehenden Datensatz mit PRIOR
 select level, ename, prior ename manager
   from emp
  start with mgr is null
connect by prior empno = mgr;

-- Abfrage zur Ermittlung der Abteilungsleiter
 select ename, deptno
   from emp
  start with mgr is null
connect by prior empno = mgr
    and prior deptno != deptno;

-- Abfragen mit der Pseudospalte LEVEL
-- Maximale Anzahl Gliederungsebenen
 select max(level) max_level
   from emp
  start with mgr is null
connect by prior empno = mgr;

-- Anzahl Mitarbeiter und Durchschnittsgehalt pro Ebene
 select level, avg(sal) avg_sal, count(*) anzahl
   from emp
  start with mgr is null
connect by prior empno = mgr
  group by level;

-- Zeige nur Mitarbeiter in Level >= 3
 select level, lpad('.', (level - 1) * 2, '.') || ename name
   from emp
  where level >= 3
  start with mgr is null
connect by prior empno = mgr;

-- Hierarchische Abfrage mit falscher Sortierung
 select level, lpad('.', (level - 1) * 2, '.') || ename name
   from emp
  start with mgr is null
connect by prior empno = mgr
  order by ename;
  
-- Hierarchische Abfrage mit korrigierter Sortierung
 select level, lpad('.', (level - 1) * 2, '.') || ename name
   from emp
  start with mgr is null
connect by prior empno = mgr
  order siblings by ename;
  
-- Erwetierung: Pseudospalte CONNECT_BY_ISLEAF
 select level, ename, connect_by_isleaf
   from emp
  start with mgr is null
connect by prior empno = mgr
  order siblings by ename;
  
-- Filterung über die Pseudospalte CONNECT_BY_IS_LEAF
 select level, ename, connect_by_isleaf
   from emp
  where connect_by_isleaf = 0
  start with mgr is null
connect by prior empno = mgr
  order siblings by ename;

-- Verwendung der Pseudospalte CONNECT_BY_ISCYCLE
update emp
   set mgr = 7369
 where empno = 7839;
 
 select level, ename
   from emp
  start with empno = 7839
connect by prior empno = mgr
  order siblings by ename;
 
 select level, ename, connect_by_iscycle
   from emp
  start with empno = 7839
connect by nocycle prior empno = mgr
  order siblings by ename;
  
rollback;

-- Verwendung des Operators CONNECT_BY_ROOT
select level, ename, connect_by_root ename manager
  from emp
  start with job = 'MANAGER'
connect by prior empno = mgr
  order siblings by ename;
  
-- Verwendung der Funktion SYS_CONNECT_BY_PATH
select level, ename,
       sys_connect_by_path(ename, '/') pfad
  from emp
  start with mgr is null
connect by prior empno = mgr
  order siblings by ename;
  
select e.ename
  from (select ename,
               sys_connect_by_path(ename, '|') path
          from emp
         where ename = 'ADAMS'
          start with mgr is null
        connect by prior empno = mgr) p,
       emp e
 where instr(p.path, e.ename) > 0
   and p.ename != e.ename;
          
-- Rekursive WITH-Klausel
with hierarchie (lvl, ename, empno) as (
     select 1 lvl, ename, empno
       from emp 
      where mgr is null
      union all 
     select h.lvl + 1, e.ename, e.empno
       from hierarchie h, emp e
      where h.empno = e.mgr) 
select lvl, ename
  from hierarchie;

-- Rekursive WITH-Klausel mit DEPTH-FIRST-Ansatz
with hierarchie (lvl, ename, empno) as (
     select 1 lvl, ename, empno
       from emp 
      where mgr is null
      union all 
     select h.lvl + 1, e.ename, e.empno
       from hierarchie h, emp e
      where h.empno = e.mgr) 
search depth first by ename set sort_seq
select lvl, ename, sort_seq
  from hierarchie
 order by sort_seq;
 
-- Rekursive WITH-Klausel mit DEPTH-FIRST-Ansatz
with hierarchie (lvl, ename, empno) as (
     select 1 lvl, ename, empno
       from emp 
      where mgr is null
      union all 
     select h.lvl + 1, e.ename, e.empno
       from hierarchie h, emp e
      where h.empno = e.mgr) 
search depth first by ename set sort_seq
select lvl, 
       lpad('.', (lvl - 1) * 2, '.') || ename name
  from hierarchie
 order by sort_seq;
 
-- Simulation der Funktionen CONNECT_BY_ROOT und SYS_CONNECT_BY_PATH
with hierarchie (lvl, ename, empno, pfad, root) as (
     select 1 lvl, ename, empno,
            '/' || ename pfad, ename root
       from emp 
      where mgr is null
      union all 
     select h.lvl + 1, e.ename, e.empno,
            h.pfad || '/' || e.ename, h.root
       from hierarchie h, emp e
      where h.empno = e.mgr) 
search depth first by ename set sort_seq
select lvl, ename, pfad, root
  from hierarchie
 order by sort_seq;
 
-- Simulation der Funktion IS_CYCLE
update emp
   set mgr = 7369
 where empno = 7839;
 
with hierarchie (lvl, ename, empno) as (
     select 1 lvl, ename, empno
       from emp 
      where empno = 7839
      union all 
     select h.lvl + 1, e.ename, e.empno
       from hierarchie h, emp e
      where h.empno = e.mgr) 
search depth first by ename set sort_seq
cycle ename set is_cycle to 1 default 0
select lvl, ename, sort_seq, is_cycle
  from hierarchie
 order by sort_seq;
 
rollback;

-- Simulation der Funktion IS_LEAF
with hierarchie (lvl, ename, empno) as (
     select 1 lvl, ename, empno
       from emp 
      where mgr is null
      union all 
     select h.lvl + 1, e.ename, e.empno
       from hierarchie h, emp e
      where h.empno = e.mgr) 
search depth first by ename set sort_seq
select lvl, ename, sort_seq,
       case when lvl - lead(lvl) over (order by sort_seq) < 0
       then 0
       else 1 end is_leaf
  from hierarchie
 order by sort_seq;
 
with hierarchie (lvl, ename, empno, is_leaf) as (
     select 1 lvl, ename, empno,
            (select decode(count(*), 0, 1, 0)
               from emp
              where mgr = m.empno) is_leaf
       from emp m
      where mgr is null
      union all 
     select h.lvl + 1, e.ename, e.empno,
            (select decode(count(*), 0, 1, 0)
               from emp
              where mgr = e.empno) is_leaf
       from hierarchie h, emp e
      where h.empno = e.mgr) 
search depth first by ename set sort_seq
select lvl, ename, is_leaf
  from hierarchie;
